/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
import { pathMode } from "./";
import { ErrorReporter } from "../error-reporter-base";
/**
 * Constructor options
 *
 * @export
 * @interface Options
 */
export interface ErrorInfoOptions {
    rawError: Error;
    /**
     * The package name, version, branch to be used in error reports
     *
     * @type {PackageInfo}
     */
    packageInfo?: PackageInfo;
    /**
     * The type of error being reported (fatal, non-fatal, main process, renderer process, etc.)
     *
     * @type {string}
     */
    errorType?: string;
    /**
     * The exe name that will be used in error reports. defaults to the real one if not provided.
     *
     * @type {string}
     */
    exeName?: string;
    errorReporter?: ErrorReporter;
    mode?: pathMode;
}
export declare class ErrorInformation {
    /**
     * Creates an instance of ErrorInformation.
     *
     * @param {Error} rawError - the Error object to be processed
     * @param {Options} [options] - other optional information. see interface
     */
    constructor(options: ErrorInfoOptions);
    /**
     * converts the raw error's stack string to a OO representation.
     *
     * @returns {StructuredError}
     */
    getStructuredStack(): StructuredError;
    /**
     * Gets a stack trace very similar to the default format, but with full paths stripped out.
     *
     * @returns {string}
     */
    getStandardStack(): string;
    /**
     * Gets a detailed stack trace including what package and version each frame came from.
     * Has full paths stripped out.
     *
     * @returns {string}
     */
    getVerboseStack(): string;
    errorName: string;
    errorMessage: string;
    errorType: string;
    rawError: Error;
    packageInfo: PackageInfo;
    exeName: string;
    errorReporter: ErrorReporter;
}
/**
 * An object representation of a stack trace
 *
 * @export
 * @interface StructuredError
 */
export interface StructuredError {
    ErrorType: string;
    ErrorMessage?: string;
    Stack: StackFrame[];
}
/**
 * An object representation of a single stack frame.
 *
 * @export
 * @interface StackFrame
 */
export interface StackFrame {
    RelativePath: string;
    FunctionName?: string;
    line?: number;
    column?: number;
    Package?: PackageInfo;
    extra?: ExtraFrameInfo;
}
/**
 * Information about a javascript package.
 *
 * @export
 * @interface PackageInfo
 */
export interface PackageInfo {
    packageJsonFullPath?: string;
    name: string;
    version?: string;
    branch?: string;
}
/**
 * Extra information that may be present about a stack frame.
 *
 * @export
 * @interface ExtraFrameInfo
 */
export interface ExtraFrameInfo {
    isNew?: boolean;
    hasAs?: string;
    isEval?: boolean;
    failedToParse?: boolean;
    rawString?: string;
}
/**
 * Sets the length of stack traces. The default is 10 frames. You can pass infinity to get full traces.
 *
 * @export
 * @param {number} limit
 */
export declare function setStackTraceLimit(limit: number): void;
/**
 * Takes an object that may or may not be an error and, if it is not an error, wraps it in an error object.
 *
 * @export
 * @param {*} e
 * @returns {Error}
 */
export declare function coerceToError(e: any): Error;
/**
 * A class that wraps non error objects so we can work with them.
 */
export declare class NonError extends Error {
    originalObject: any;
    constructor(input: any);
}
/**
 * shorthand for using a value or a default
 *
 * @export
 * @template T
 * @param {T} defaultValue - the default value
 * @param {T} actual - the actual value
 * @param {boolean} [strict=true] - whether to allow falsy values other than undefined.
 * If false, the default will override any falsy value.
 * If true, the default will be used only if the actual is undefined
 * @returns {T}
 */
export declare function defaultValue<T>(defaultValue: T, actual: T, strict?: boolean): T;
